// ==UserScript==
// @name         5ch 投稿欄 iframe化 （ページ偏移対策）
// @namespace    http://example.com/
// @version      3.8.1
// @description  .formbox を iframe 化。投稿後のURL変化・戻りや遅延表示にも完全対応。高さ自動調整機能付き。
// @match        *://*.5ch.net/test/read.cgi/*
// @grant        none
// ==/UserScript==

(function () {
    'use strict';

    const isInIframe = window.top !== window.self;
    const parentUrl = document.referrer;
    const CHECK_INTERVAL = 1000;
    const HEIGHT_PADDING = 20;
    const MAX_HEIGHT = 1000;
    const FIXED_OTHER_HEIGHT = 800;

    function getThreadPath(url) {
        try {
            const u = new URL(url);
            const match = u.pathname.match(/^\/test\/read\.cgi\/[^\/]+\/\d+/);
            return match ? match[0] : '';
        } catch {
            return '';
        }
    }

    if (!isInIframe) {
        // 親ページ側処理
        const formBox = document.querySelector('.formbox');
        if (!formBox) return;

        const iframe = document.createElement('iframe');
        iframe.id = 'submit';
        iframe.name = 'submit';
        iframe.src = window.location.href;
        iframe.style.margin = "40px 0 0 0";
        iframe.style.width = '100%';
        iframe.style.border = 'none';
        iframe.style.overflow = 'hidden';
        iframe.style.height = '50px'; // 初期高さ

        window.addEventListener('message', (event) => {
            if (event.data && event.data.type === 'resize-iframe') {
                let paddedHeight = event.data.height + HEIGHT_PADDING;
                if (paddedHeight > MAX_HEIGHT) paddedHeight = MAX_HEIGHT;
                if (iframe.style.height !== `${paddedHeight}px`) {
                    iframe.style.height = `${paddedHeight}px`;
                }
            }
        });

        formBox.parentNode.replaceChild(iframe, formBox);

    } else {
        // iframe 内処理
        let lastUrl = location.href;
        let lastHeight = 0;
        let observer = null;

        // 履歴APIの監視を追加
        (function() {
            const origPushState = history.pushState;
            const origReplaceState = history.replaceState;

            function onStateChange() {
                waitForFormBoxAndHandleContent();
            }

            history.pushState = function(...args) {
                const ret = origPushState.apply(this, args);
                onStateChange();
                return ret;
            };

            history.replaceState = function(...args) {
                const ret = origReplaceState.apply(this, args);
                onStateChange();
                return ret;
            };

            window.addEventListener('popstate', () => {
                waitForFormBoxAndHandleContent();
            });
        })();

        function isolateFormIfNeeded() {
            const formBox = document.querySelector('.formbox');
            if (!formBox) return false;

            const threadPathCurrent = getThreadPath(location.href);
            const threadPathParent = getThreadPath(parentUrl);

            // フォームがあるならまずフォームだけ切り出す
            const onlyFormBoxExists =
                document.body.children.length === 1 &&
                document.body.firstElementChild === formBox;

            if (!onlyFormBoxExists) {
                document.body.innerHTML = '';
                document.body.style.margin = '0';
                document.documentElement.style.margin = '0';
                document.documentElement.style.overflow = 'hidden';
                document.body.appendChild(formBox);
            }

            // スレッドURLと違う場合はフォームがあっても高さ固定モードに
            if (threadPathCurrent !== threadPathParent) {
                return false;
            }
            return true;
        }

        function sendHeight(height) {
            if (Math.abs(height - lastHeight) > 1) {
                lastHeight = height;
                parent.postMessage({ type: 'resize-iframe', height }, '*');
            }
        }

        function observeFormBox(formBox) {
            if (observer) observer.disconnect();

            observer = new MutationObserver(() => {
                setTimeout(() => {
                    const height = formBox.offsetHeight;
                    const cappedHeight = Math.min(height, MAX_HEIGHT);
                    sendHeight(cappedHeight);
                }, 100);
            });

            observer.observe(formBox, {
                childList: true,
                subtree: true,
                attributes: true
            });
        }

        function handleContent(formBox) {
            if (isolateFormIfNeeded()) {
                if (!formBox) return;
                observeFormBox(formBox);
                const initialHeight = Math.min(formBox.offsetHeight, MAX_HEIGHT);
                sendHeight(initialHeight);
            } else {
                if (observer) {
                    observer.disconnect();
                    observer = null;
                }
                sendHeight(FIXED_OTHER_HEIGHT);
            }
        }

        function waitForFormBoxAndHandleContent() {
            let tries = 0;
            const maxTries = 20;
            const interval = 1000;

            function check() {
                const formBox = document.querySelector('.formbox');
                if (formBox) {
                    handleContent(formBox);
                } else if (tries < maxTries) {
                    tries++;
                    setTimeout(check, interval);
                } else {
                    // 一定時間待ってもフォームなければ高さ固定
                    if (observer) {
                        observer.disconnect();
                        observer = null;
                    }
                    sendHeight(FIXED_OTHER_HEIGHT);
                }
            }
            check();
        }

        // 初回処理
        waitForFormBoxAndHandleContent();

        // URL変更監視（投稿後など）
        setInterval(() => {
            if (location.href !== lastUrl) {
                lastUrl = location.href;
                waitForFormBoxAndHandleContent();
            }
        }, CHECK_INTERVAL);

        // DOM再構築監視（遅延読込対応）
        document.addEventListener('DOMContentLoaded', () => {
            waitForFormBoxAndHandleContent();
        });

        // リサイズ時の高さ調整
        window.addEventListener('resize', () => {
            const formBox = document.querySelector('.formbox');
            if (observer && formBox) {
                const height = Math.min(formBox.offsetHeight, MAX_HEIGHT);
                sendHeight(height);
            } else {
                sendHeight(FIXED_OTHER_HEIGHT);
            }
        });
    }

    // フォーム内テキスト色の指定
    (function() {
        'use strict';
        const style = document.createElement('style');
        style.innerHTML = `
        input.formelem,
        textarea.formelem {
            color: #e0e0e0 !important;
            /* background-color: #111 !important; */
        }

        input.formelem::placeholder,
        textarea.formelem::placeholder {
            color: #e0e0e0 !important;
        }
    `;
        document.head.appendChild(style);
    })();

})();
